/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.store;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import org.neo4j.helpers.collection.Pair;
import org.neo4j.kernel.impl.store.DynamicArrayStore;
import org.neo4j.kernel.impl.store.InvalidRecordException;
import org.neo4j.kernel.impl.store.PropertyType;
import org.neo4j.kernel.impl.store.record.PropertyBlock;
import org.neo4j.values.storable.ArrayValue;
import org.neo4j.values.storable.CoordinateReferenceSystem;
import org.neo4j.values.storable.FloatingPointArray;
import org.neo4j.values.storable.PointValue;
import org.neo4j.values.storable.Value;
import org.neo4j.values.storable.Values;

public enum GeometryType {
    GEOMETRY_INVALID(0, "Invalid"){

        @Override
        public Value decode(CoordinateReferenceSystem crs, int dimension, long[] valueBlocks, int offset) {
            throw new UnsupportedOperationException("Cannot decode invalid geometry");
        }

        @Override
        public int calculateNumberOfBlocksUsedForGeometry(long firstBlock) {
            return -1;
        }

        @Override
        public ArrayValue decodeArray(GeometryHeader header, byte[] data) {
            throw new UnsupportedOperationException("Cannot decode invalid geometry array");
        }
    }
    ,
    GEOMETRY_POINT(1, "Point"){

        @Override
        public Value decode(CoordinateReferenceSystem crs, int dimension, long[] valueBlocks, int offset) {
            double[] coordinate = new double[dimension];
            for (int i = 0; i < dimension; ++i) {
                coordinate[i] = Double.longBitsToDouble(valueBlocks[i + 1 + offset]);
            }
            return Values.pointValue(crs, coordinate);
        }

        @Override
        public int calculateNumberOfBlocksUsedForGeometry(long firstBlock) {
            int dimension = GeometryType.getDimension(firstBlock);
            if (dimension > GeometryType.getMaxSupportedDimensions()) {
                return -1;
            }
            return 1 + dimension;
        }

        @Override
        public ArrayValue decodeArray(GeometryHeader header, byte[] data) {
            byte[] dataHeader = PropertyType.ARRAY.readDynamicRecordHeader(data);
            byte[] dataBody = new byte[data.length - dataHeader.length];
            System.arraycopy(data, dataHeader.length, dataBody, 0, dataBody.length);
            Value dataValue = DynamicArrayStore.getRightArray((Pair<byte[], byte[]>)Pair.of((Object)dataHeader, (Object)dataBody));
            if (dataValue instanceof FloatingPointArray) {
                FloatingPointArray numbers = (FloatingPointArray)dataValue;
                PointValue[] points = new PointValue[numbers.length() / header.dimension];
                for (int i = 0; i < points.length; ++i) {
                    double[] coords = new double[header.dimension];
                    for (int d = 0; d < header.dimension; ++d) {
                        coords[d] = numbers.doubleValue(i * header.dimension + d);
                    }
                    points[i] = Values.pointValue(header.crs, coords);
                }
                return Values.pointArray(points);
            }
            throw new InvalidRecordException("Point array with unexpected type. Actual:" + dataValue.getClass().getSimpleName() + ". Expected: FloatingPointArray.");
        }
    };

    private static final GeometryType[] TYPES;
    private static final Map<String, GeometryType> all;
    private static final long GEOMETRY_TYPE_MASK = 0xF0000000L;
    private static final long DIMENSION_MASK = 0xF00000000L;
    private static final long CRS_TABLE_MASK = 0xF000000000L;
    private static final long CRS_CODE_MASK = 0xFFFF0000000000L;
    private static final long PRECISION_MASK = 0x100000000000000L;
    private final int gtype;
    private final String name;

    private static int getGeometryType(long firstBlock) {
        return (int)((firstBlock & 0xF0000000L) >> 28);
    }

    private static int getDimension(long firstBlock) {
        return (int)((firstBlock & 0xF00000000L) >> 32);
    }

    private static int getCRSTable(long firstBlock) {
        return (int)((firstBlock & 0xF000000000L) >> 36);
    }

    private static int getCRSCode(long firstBlock) {
        return (int)((firstBlock & 0xFFFF0000000000L) >> 40);
    }

    private static boolean isFloatPrecision(long firstBlock) {
        return (firstBlock & 0x100000000000000L) >> 56 == 1L;
    }

    private static int getMaxSupportedDimensions() {
        return PropertyType.getPayloadSizeLongs() - 1;
    }

    public static int calculateNumberOfBlocksUsed(long firstBlock) {
        GeometryType geometryType = GeometryType.find(GeometryType.getGeometryType(firstBlock));
        return geometryType.calculateNumberOfBlocksUsedForGeometry(firstBlock);
    }

    private static GeometryType find(int gtype) {
        if (gtype < TYPES.length && gtype >= 0) {
            return TYPES[gtype];
        }
        return GEOMETRY_INVALID;
    }

    public static Value decode(PropertyBlock block) {
        return GeometryType.decode(block.getValueBlocks(), 0);
    }

    public static Value decode(long[] valueBlocks, int offset) {
        long firstBlock = valueBlocks[offset];
        int gtype = GeometryType.getGeometryType(firstBlock);
        int dimension = GeometryType.getDimension(firstBlock);
        if (GeometryType.isFloatPrecision(firstBlock)) {
            throw new UnsupportedOperationException("Float precision is unsupported in Geometry properties");
        }
        if (dimension > GeometryType.getMaxSupportedDimensions()) {
            throw new UnsupportedOperationException("Points with more than " + GeometryType.getMaxSupportedDimensions() + " dimensions are not supported");
        }
        CoordinateReferenceSystem crs = CoordinateReferenceSystem.get(GeometryType.getCRSTable(firstBlock), GeometryType.getCRSCode(firstBlock));
        return GeometryType.find(gtype).decode(crs, dimension, valueBlocks, offset);
    }

    public static long[] encodePoint(int keyId, CoordinateReferenceSystem crs, double[] coordinate) {
        if (coordinate.length > GeometryType.getMaxSupportedDimensions()) {
            throw new UnsupportedOperationException("Points with more than " + GeometryType.getMaxSupportedDimensions() + " dimensions are not supported");
        }
        int idBits = 24;
        long keyAndType = (long)keyId | (long)PropertyType.GEOMETRY.intValue() << idBits;
        long gtypeBits = GeometryType.GEOMETRY_POINT.gtype << idBits + 4;
        long dimensionBits = (long)coordinate.length << idBits + 8;
        long crsTableIdBits = (long)crs.getTable().getTableId() << idBits + 12;
        long crsCodeBits = (long)crs.getCode() << idBits + 16;
        long[] data = new long[1 + coordinate.length];
        data[0] = keyAndType | gtypeBits | dimensionBits | crsTableIdBits | crsCodeBits;
        for (int i = 0; i < coordinate.length; ++i) {
            data[1 + i] = Double.doubleToLongBits(coordinate[i]);
        }
        return data;
    }

    public static byte[] encodePointArray(PointValue[] points) {
        int dimension = points[0].coordinate().length;
        CoordinateReferenceSystem crs = points[0].getCoordinateReferenceSystem();
        for (int i = 1; i < points.length; ++i) {
            if (dimension != points[i].coordinate().length) {
                throw new IllegalArgumentException("Attempting to store array of points with inconsistent dimension. Point " + i + " has a different dimension.");
            }
            if (crs.equals(points[i].getCoordinateReferenceSystem())) continue;
            throw new IllegalArgumentException("Attempting to store array of points with inconsistent CRS. Point " + i + " has a different CRS.");
        }
        double[] data = new double[points.length * dimension];
        for (int i = 0; i < data.length; ++i) {
            data[i] = points[i / dimension].coordinate()[i % dimension];
        }
        GeometryHeader geometryHeader = new GeometryHeader(GeometryType.GEOMETRY_POINT.gtype, dimension, crs);
        byte[] bytes2 = DynamicArrayStore.encodeFromNumbers(data, 6);
        geometryHeader.writeArrayHeaderTo(bytes2);
        return bytes2;
    }

    public static ArrayValue decodeGeometryArray(GeometryHeader header, byte[] data) {
        return GeometryType.find(header.geometryType).decodeArray(header, data);
    }

    private GeometryType(int gtype, String name) {
        this.gtype = gtype;
        this.name = name;
    }

    public abstract Value decode(CoordinateReferenceSystem var1, int var2, long[] var3, int var4);

    public abstract int calculateNumberOfBlocksUsedForGeometry(long var1);

    public abstract ArrayValue decodeArray(GeometryHeader var1, byte[] var2);

    public int getGtype() {
        return this.gtype;
    }

    public String getName() {
        return this.name;
    }

    static {
        TYPES = GeometryType.values();
        all = new HashMap<String, GeometryType>(TYPES.length);
        for (GeometryType geometryType : TYPES) {
            all.put(geometryType.name, geometryType);
        }
    }

    public static class GeometryHeader {
        private final int geometryType;
        private final int dimension;
        private final CoordinateReferenceSystem crs;

        private GeometryHeader(int geometryType, int dimension, CoordinateReferenceSystem crs) {
            this.geometryType = geometryType;
            this.dimension = dimension;
            this.crs = crs;
        }

        private GeometryHeader(int geometryType, int dimension, int crsTableId, int crsCode) {
            this(geometryType, dimension, CoordinateReferenceSystem.get(crsTableId, crsCode));
        }

        private void writeArrayHeaderTo(byte[] bytes2) {
            bytes2[0] = (byte)PropertyType.GEOMETRY.intValue();
            bytes2[1] = (byte)this.geometryType;
            bytes2[2] = (byte)this.dimension;
            bytes2[3] = (byte)this.crs.getTable().getTableId();
            bytes2[4] = (byte)((long)(this.crs.getCode() >> 8) & 0xFFL);
            bytes2[5] = (byte)((long)this.crs.getCode() & 0xFFL);
        }

        static GeometryHeader fromArrayHeaderBytes(byte[] header) {
            int geometryType = Byte.toUnsignedInt(header[1]);
            int dimension = Byte.toUnsignedInt(header[2]);
            int crsTableId = Byte.toUnsignedInt(header[3]);
            int crsCode = (Byte.toUnsignedInt(header[4]) << 8) + Byte.toUnsignedInt(header[5]);
            return new GeometryHeader(geometryType, dimension, crsTableId, crsCode);
        }

        public static GeometryHeader fromArrayHeaderByteBuffer(ByteBuffer buffer) {
            int geometryType = Byte.toUnsignedInt(buffer.get());
            int dimension = Byte.toUnsignedInt(buffer.get());
            int crsTableId = Byte.toUnsignedInt(buffer.get());
            int crsCode = (Byte.toUnsignedInt(buffer.get()) << 8) + Byte.toUnsignedInt(buffer.get());
            return new GeometryHeader(geometryType, dimension, crsTableId, crsCode);
        }
    }
}

